import { Button } from "~/components/ui/button";
import AuthLayout from "../layout/Auth";
import { useForm, SubmitHandler, FormProvider } from "react-hook-form";
import Heading from "~/components/core/Text/Heading";
import FormElement from "~/components/core/FormElements/FormElement";
import { LoginProps } from "~/store/features/auth/type";
import { Link, useNavigate } from "@remix-run/react";
import { useForgotPasswordMutation } from "~/store/features/auth/signup";
import { toast } from "~/hooks/use-toast";
import { CircleCheck, TriangleAlert } from "lucide-react";
import { useState } from "react";
import { Dialog } from "~/components/ui/dialog";

const AuthHeading = {
  title: "",
  subtitle: "",
};

const MainHeading = {
  title: "Reset Password",
  subtitle: "A reset link will be sent to your email",
};

const ResetPassword = () => {
  const navigate = useNavigate();

  const [isOpen, setIsOpen] = useState(false);
  const [email, setEmail] = useState("");

  const methods = useForm<LoginProps>();

  const { handleSubmit } = methods;

  const [forgotPassword, { isLoading }] = useForgotPasswordMutation();

  const onSubmit: SubmitHandler<LoginProps> = (data) => {
    forgotPassword(data)
      .unwrap()
      .then((res) => {
        if (res?.success) {
          setEmail(data?.email);
          setIsOpen(true);
        }
      })
      .catch((err) => {
        if (err?.data?.data?.non_field_errors) {
          toast({
            title: "An error occurred",
            description: err?.data?.data?.non_field_errors[0],
            variant: "destructive",
            icon: (
              <TriangleAlert
                color="red"
                size={35}
                className="bg-red-100 p-1 rounded"
              />
            ),
          });
        } else if (err?.data?.data) {
          toast({
            title: "An error occurred",
            description: err?.data?.data?.username[0]
              ? err?.data?.data?.username[0]
              : err?.data?.data?.email[0],
            variant: "destructive",
            icon: (
              <TriangleAlert
                color="red"
                size={35}
                className="bg-red-100 p-1 rounded"
              />
            ),
          });
        } else {
          toast({
            title: "An error occurred",
            description: "Something went wrong, please try again",
            variant: "destructive",
            icon: (
              <TriangleAlert
                color="red"
                size={35}
                className="bg-red-100 p-1 rounded"
              />
            ),
          });
        }
        // console.log(err?.data?.data?.non_field_errors[0]);
      });
  };

  return (
    <AuthLayout title={AuthHeading.title} subtitle={AuthHeading.subtitle}>
      <FormProvider {...methods}>
        <form
          className="flex flex-col gap-5 w-full"
          onSubmit={handleSubmit(onSubmit)}
        >
          <Heading title={MainHeading.title} subtitle={MainHeading.subtitle} />
          <FormElement type="email" fieldName="email" required />
          <Button size="lg" text="Send Reset Link" isLoading={isLoading} />
        </form>
        <p className="flex gap-2 justify-center items-center font-semibold my-4">
          Not you?
          <Link to="/user/login" className="text-primary">
            Log in
          </Link>
        </p>
      </FormProvider>
      <Dialog isOpen={isOpen}>
        <div className="space-y-4">
          <CircleCheck
            color="green"
            size={35}
            className="bg-green-100 p-2 rounded-lg absolute top-6"
          />
          <div>
            <h2 className="text-2xl font-semibold">Reset password link sent</h2>
            <p className="text-sm text-muted-foreground">
              A reset passsword link has been sent to {email}
            </p>
          </div>
          <hr />
          <Button
            variant="default"
            className="w-full"
            onClick={() => {
              navigate("/user/login");
            }}
            text="Go to login"
          />
        </div>
      </Dialog>
    </AuthLayout>
  );
};

export default ResetPassword;
