import { useEffect, useState } from "react";
import type { MetaFunction } from "@remix-run/node";
import DefaultLayout from "~/layout/Default";
import { Button } from "~/components/ui/button";
import { Clock, FileText, Truck } from "lucide-react";
import {
  bgImages,
  logos,
  productCategory,
  products,
  testimonial,
} from "./data";
import Header from "./section";
import Category from "./category";
import BestProduct from "./best-product";
import Section from "./section";
import { images } from "~/assets";
import SimpleCard from "./simple-card";
import Testimonial from "./testimonial";

export const meta: MetaFunction = () => {
  return [
    { title: "New Remix App" },
    { name: "description", content: "Welcome to Remix!" },
  ];
};

export default function Index() {
  const [currentImage, setCurrentImage] = useState(0);

  useEffect(() => {
    const interval = setInterval(() => {
      setCurrentImage((prevImage) => (prevImage + 1) % bgImages.length);
    }, 5000);

    return () => clearInterval(interval);
  }, [bgImages?.length]);

  return (
    <DefaultLayout>
      <section className="flex flex-col">
        <div
          className="relative bg-auto w-full h-screen py-12 px-3 flex items-end"
          style={{
            backgroundImage: `url(${bgImages[currentImage].url})`,
            backgroundPosition: bgImages[currentImage].position,
            backgroundSize: bgImages[currentImage].size,
            backgroundRepeat: "no-repeat",
            backgroundColor: bgImages[currentImage].color,
            filter: "grayscale(30%)",
          }}
        >
          <div className="bg-white bg-opacity-10 backdrop-blur-md backdrop-contrast-100 flex flex-col gap-3 max-w-[37rem] p-6 rounded-md shadow-sm absolute left-12 bottom-36">
            <h2 className="bg-white bg-opacity-30 backdrop-blur-sm rounded-full font-semibold w-fit text-gray-700 px-3 py-1">
              Create Products That Inspire.
            </h2>
            <h1 className="font-semibold text-6xl leading-tight text-white">
              Make It Yours, Make it Memorable
            </h1>
            <p className="my-3 text-lg font-semibold text-gray-700">
              With endless options and fast delivery, creating custom products
              has never been easier.
            </p>
            <Button
              className="py-6 text-md px-8 w-fit"
              text="Start Customizing"
            />
          </div>
        </div>
        <div className="flex justify-evenly bg-[#F5F5F5] py-3.5">
          <div className="flex items-center gap-3 font-semibold text-lg">
            <Clock size={40} />1 week turn around
          </div>
          <div className="flex items-center gap-3 font-semibold text-lg">
            <FileText size={40} />
            Printed in America
          </div>
          <div className="flex items-center gap-3 font-semibold text-lg">
            <Truck size={40} />
            Free Delivery <span className="text-sm -mx-1.5">over $75</span>
          </div>
        </div>
      </section>
      <div className="flex flex-col gap-24 py-14">
        <section className="flex flex-col gap-12 overflow-hidden">
          <div className="relative text-center text-sm after:absolute after:inset-0 after:top-1/2 after:z-0 after:flex after:items-center after:border-t after:border-border">
            <span className="relative z-10 bg-background px-2 text-black text-xl font-semibold">
              Trusted by over 1,000,000 Customers
            </span>
          </div>
          <div className="flex gap-24 animate-scroll whitespace-nowrap">
            {logos.map((logo, index) => (
              <img
                key={index}
                src={logo}
                alt={`Trusted brand ${index + 1}`}
                className="h-16 w-auto"
              />
            ))}
            {logos.map((logo, index) => (
              <img
                key={`duplicate-${index}`}
                src={logo}
                alt={`Trusted brand duplicate ${index + 1}`}
                className="h-16 w-auto"
              />
            ))}
          </div>
          <hr />
        </section>
        <Section heading="Explore Categories">
          <Category productCategory={productCategory} />
        </Section>
        <Section heading="Best Selling Products">
          <div className="grid grid-cols-3 gap-x-6 gap-y-8">
            {products?.map((product, index) => (
              <BestProduct
                key={index}
                image={product.image}
                name={product.name}
                price={product.price}
              />
            ))}
          </div>
          <Button
            text="View More"
            className="w-full max-w-60 py-6 text-lg mx-auto mt-8"
          />
        </Section>
        <Section heading="Shop Featured Brands">
          <div className="flex justify-between flex-wrap gap-4">
            {Array.from({ length: 12 }, (_, index) => (
              <div key={index} className="min-w-40">
                <img
                  src={images[`brand${index + 1}`]}
                  alt={`Brand ${index + 1}`}
                />
              </div>
            ))}
          </div>
        </Section>
        <Section heading="Why Clients Choose Us">
          <div className="grid grid-cols-2 gap-5">
            <SimpleCard
              heading="Quality"
              color="bg-[#F0F5F9]"
              content="Our products are made to stand out and last. We focus on exceptional materials and meticulous printing methods to ensure that each item meets the highest standards of durability and aesthetics."
            />
            <SimpleCard
              heading="Flexibility"
              color="bg-[#FDBA48]"
              content="We provide you with complete creative control—from product selection to color and design. Our intuitive tools make it easy to bring your unique vision to life, however complex or simple it may be."
            />
          </div>
          <div className="grid grid-cols-3 gap-5">
            <SimpleCard
              heading="Reliability"
              color="bg-[#EAE8ED]"
              content="Your time and satisfaction are our priorities. From design to doorstep, we make sure each order is handled with accuracy, care, and efficiency, so you can count on dependable service every time."
            />
            <SimpleCard
              heading="Support"
              color="bg-[#BCEDED]"
              content="Our customer service team is here for you, whether you need help with design choices, placing an order, or tracking a shipment. We’re committed to assisting you at every step until you’re completely satisfied."
            />
            <SimpleCard
              heading="Value"
              color="bg-[#EAE8ED]"
              content="Get premium quality without the premium price. We’re committed to offering fair pricing, whether you need a single item or a bulk order, so you never have to compromise on quality or budget."
            />
          </div>
        </Section>
        <Section heading="What Our Customers Say">
          <Testimonial testimonial={testimonial} />
        </Section>
      </div>
    </DefaultLayout>
  );
}
