import { DynamicFieldErrorProps, FieldProps, FieldType } from "./FieldType";
import { ReactElement } from "react";

import TextInput from "../TextInput";
import TextArea from "../TextArea";
import PhoneNumber from "../PhoneNumber";
import SelectInput from "../SelectInput";
import RadioInput from "../RadioInput";
import CheckBox from "../CheckBox";
import StarRating from "../StarRating";

export const getFieldType = ({
  register,
  control,
  fieldName,
  fieldLabel,
  type,
  required,
  isValidated,
  validateFn,
  onChangeFn,
  handleKeyDown,
  defaultValue,
  placeholder,
  isDisabled,
  maxLength,
  optionsData = [],
  radioOptions = [],
  withDescription,
  defaultChecked,
  tooltip,
  jsxFieldLabel,
  moreCss,
}: FieldProps): FieldType => {
  let content: ReactElement | null = null;

  switch (type) {
    case "text":
    case "number":
    case "date":
    case "password":
    case "email":
      content = (
        <TextInput
          type={type}
          fieldName={fieldName}
          fieldLabel={fieldLabel}
          register={register}
          required={required}
          isValidated={isValidated}
          validateFn={validateFn}
          handleKeyDown={handleKeyDown}
          defaultValue={defaultValue}
          placeholder={placeholder}
          isDisabled={isDisabled}
          tooltip={tooltip}
          onChangeFn={onChangeFn}
          moreCss={moreCss}
        />
      );
      break;

    case "textarea":
      content = (
        <TextArea
          register={register}
          fieldName={fieldName}
          fieldLabel={fieldLabel}
          required={required}
          isValidated={isValidated}
          validateFn={validateFn}
          handleKeyDown={handleKeyDown}
          defaultValue={defaultValue}
          isDisabled={isDisabled}
        />
      );
      break;

    case "phone":
      content = (
        <PhoneNumber
          moreCss={moreCss}
          control={control}
          fieldName={fieldName}
          maxLength={maxLength}
          required={required}
          isDisabled={isDisabled}
          defaultValue={defaultValue}
        />
      );
      break;

    case "select":
      content = (
        <SelectInput
          control={control}
          fieldName={fieldName}
          fieldLabel={fieldLabel}
          defaultValue={defaultValue}
          required={required}
          optionsData={optionsData}
          isDisabled={isDisabled}
          onChangeFn={onChangeFn}
          moreCss={moreCss}
        />
      );
      break;

    // case "address-selector":
    //   content = (
    //     <AddressSelector
    //       register={register}
    //       fieldName={fieldName}
    //       // defaultValue={defaultValue}
    //       required={required}
    //       setValue={setValue}
    //       registerOptions={registerOptions}
    //     />
    //   );
    //   break;

    case "star-rating":
      content = (
        <StarRating
          fieldName={fieldName}
          required={required}
          control={control}
          defaultValue={defaultValue}
        />
      );
      break;

    case "radio-options":
      content = (
        <RadioInput
          register={register}
          fieldName={fieldName}
          required={required}
          options={radioOptions}
          withDescription={withDescription}
          defaultChecked={defaultChecked}
        />
      );
      break;

    case "checkbox":
      content = (
        <CheckBox
          register={register}
          fieldName={fieldName}
          fieldLabel={fieldLabel}
          required={required}
          defaultChecked={defaultChecked}
          jsxFieldLabel={jsxFieldLabel}
        />
      );
      break;

    default:
      break;
  }

  return { content, type };
};

export const dynamicFormFieldError = ({
  dynamic,
  errors,
}: DynamicFieldErrorProps): string | null => {
  let errorMessage = "";
  if (dynamic && errors) {
    if (dynamic.grandParent) {
      if (
        dynamic.grandParentType === "array" &&
        typeof dynamic.parentIndex === "number"
      ) {
        if (dynamic.type === "array") {
          errorMessage =
            errors[dynamic.grandParent]?.[dynamic.parentIndex]?.[
              dynamic.parent
            ]?.[dynamic.index]?.[dynamic.fieldName]?.message;
        } else {
          errorMessage =
            errors[dynamic.grandParent]?.[dynamic.parentIndex]?.[
              dynamic.parent
            ]?.[dynamic.fieldName]?.message;
        }
      } else if (dynamic.grandParentType === "object") {
        if (dynamic.type === "array") {
          errorMessage =
            errors[dynamic.grandParent]?.[dynamic.parent]?.[dynamic.index]?.[
              dynamic.fieldName
            ]?.message;
        } else {
          errorMessage =
            errors[dynamic.grandParent]?.[dynamic.parent]?.[dynamic.fieldName]
              ?.message;
        }
      }
    } else {
      if (dynamic.type === "array") {
        errorMessage =
          errors[dynamic.parent]?.[dynamic.index]?.[dynamic.fieldName]?.message;
      } else {
        errorMessage = errors[dynamic.parent]?.[dynamic.fieldName]?.message;
      }
    }
    return errorMessage;
  } else {
    return null;
  }
};

export const formatReactSelectStyles: Record<
  string,
  (
    baseStyles: object,
    state: { isDisabled: string; isSelected: string }
  ) => object
> = {
  control: (baseStyles, state) => ({
    ...baseStyles,
    border: "none",
    minHeight: "fit-content",
    backgroundColor: state.isDisabled ? "transparent" : "transparent",
  }),
  valueContainer: (baseStyles) => ({
    ...baseStyles,
    padding: 0,
    border: "none",
    outline: "none",
  }),
  input: (baseStyles) => ({
    ...baseStyles,
    margin: 0,
    padding: 0,
    border: "none",
    outline: "none",
  }),
  placeholder: (baseStyles) => ({
    ...baseStyles,
    color: "#ced6de",
    fontFamily: "Open Sans",
  }),
  menu: (baseStyles) => ({
    ...baseStyles,
    marginTop: "0.5rem",
    marginLeft: "-0.75rem",
    color: "#414141",
  }),
  option: (baseStyles, state) => ({
    ...baseStyles,
    cursor: "pointer",
    backgroundColor: state.isSelected ? "#eef6f0" : "white",
    color: state.isSelected ? "#1b7232" : "#414141",
  }),
  indicatorSeparator: () => ({
    display: "none",
  }),
  dropdownIndicator: () => ({
    position: "relative",
    right: "0.2rem",
    bottom: "0.1rem",
    color: "#ced6de",
    cursor: "pointer",
  }),
  clearIndicator: () => ({
    position: "relative",
    right: "0.5rem",
    bottom: "0.1rem",
    color: "red",
    cursor: "pointer",
  }),
};
